<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor progress widget.
 *
 * Elementor widget that displays an escalating progress bar.
 *
 * @since 1.0.0
 */
class socity_contactbox_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve progress widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'socity-contactbox';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve progress widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'socity Contact Box', 'socitycore' );
	}

    
	public function get_categories() {
		return [ 'socitycore' ];
	}
    
	/**
	 * Get widget icon.
	 *
	 * Retrieve progress widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-t-letter';
	}
	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'socity', 'contact-box' ];
	}

	/**
	 * Register progress widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {
		
		$this->start_controls_section(
			'socity_contactbox_section',
			[
				'label' => esc_html__( 'Content', 'socitycore' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'socity_cbox_icon_type',
			[
			    'label' => esc_html__('Icon Type','socitycore'),
			    'type' =>Controls_Manager::CHOOSE,
			    'options' =>[
				  'img' =>[
					'title' =>esc_html__('Image','socitycore'),
					'icon' =>'fa fa-picture-o',
				  ],
				  'icon' =>[
					'title' =>esc_html__('Icon','socitycore'),
					'icon' =>'fa fa-info',
				  ]
			    ],
			    'default' => 'icon',
			]
		  );
	
		  $this->add_control(
			'socity_cbox_icon_img',
			[
			    'label' => esc_html__('Image','socitycore'),
			    'type'=>Controls_Manager::MEDIA,
			    'default' => [
				  'url' => Utils::get_placeholder_image_src(),
			    ],
			    'condition' => [
				  'socity_cbox_icon_type' => 'img',
			    ]
			]
		  );
	
		  $this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
			    'name' => 'socity_cbox_icon_img_resize',
			    'default' => 'large',
			    'separator' => 'none',
			    'condition' => [
				  'socity_cbox_icon_type' => 'img',
			    ]
			]
		  );
		  $this->add_control(
			'socity_cbox_icon',
			[
				'label' => esc_html__( 'Icon', 'socitycore' ),
				'type' => \Elementor\Controls_Manager::ICON,
				'condition'=>[
					'socity_cbox_icon_type'=> 'icon',
				],
				'label_block' => true,
			]
		);
		$this->add_control(
			'socity_cbox_title',
			[
			    'label' => esc_html__( 'Title', 'socitycore' ),
			    'type'          => Controls_Manager::TEXT,
			    'default'       => esc_html__('Contact Us','socitycore'),
			]
		);
		$this->add_control(
			'socity_cbox_content',
			[
			    'label' => esc_html__( 'content', 'socitycore' ),
			    'type' => Controls_Manager::WYSIWYG,
			    'default'       => wp_kses(
				    __('
				    <ul>
				    	<li>24info@example.com</li>
				    	<li>support@example.com</li>
				    </ul>','socitycore'),
				    array(
					'ul' => array(),
					'li' => array(),
					'strong' => array(),
					'span' => array(),
					'p' => array(),
				    )
			    )
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'socity_cbox_box_style',
			[
			    'label' => esc_html__( 'Box Style', 'socitycore' ),
			    'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_responsive_control(
			'socity_cbox_alignment',
			[
				'label' => esc_html__( 'Alignment', 'socitycore' ),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'socitycore' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'socitycore' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'socitycore' ),
						'icon' => 'fa fa-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .static-contact-box' => 'text-align: {{VALUE}};',
				],
				'default' => 'center',
				'toggle' => true,
			]
		);
		$this->add_responsive_control(
			'socity_cbox_margin',
			[
			    'label' => esc_html__( 'Margin', 'socitycore' ),
			    'type' => Controls_Manager::DIMENSIONS,
			    'size_units' => [ 'px', '%', 'em' ],
			    'selectors' => [
				  '{{WRAPPER}} .static-contact-box' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
			    ],
			    'separator' =>'before',
			]
		);
		$this->add_responsive_control(
			'socity_cbox_padding',
			[
			    'label' => esc_html__( 'Padding', 'socitycore' ),
			    'type' => Controls_Manager::DIMENSIONS,
			    'size_units' => [ 'px', '%', 'em' ],
			    'selectors' => [
				  '{{WRAPPER}} .static-contact-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
			    ],
			    'separator' =>'before',
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'socity_cbox_icons_styles',
			[
			    'label' => esc_html__( 'Icons Style', 'socitycore' ),
			    'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->start_controls_tabs('socity_cbox_icons_style_tabs');

        	$this->start_controls_tab( 'socity_cbox_ficon_tab',
			[
				'label' => esc_html__( 'Font icon Normal', 'socitycore' ),
			]
		);
		$this->add_control(
			'socity_cbox_ficon_color',
			[
				'label' => esc_html__( 'Icon Color', 'socitycore' ),
				'type' => Controls_Manager::COLOR,
				'default' => '#0b5be0',
				'selectors' => [
					'{{WRAPPER}} .cbox-icon i:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_responsive_control(
			'socity_cbox_ficon_size',
			[
				'label' => esc_html__( 'Font Icon Size', 'socitycore' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 40,
						'max' => 200,
						'step' => 1,
					],
					'%' => [
						'min' => 2,
						'max' => 500,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 64,
				],
				'selectors' => [
					'{{WRAPPER}} .cbox-icon i:before' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);
		
		$this->end_controls_tab();
		$this->start_controls_tab( 'socity_cbox_ficon_htab',
			[
				'label' => esc_html__( 'Font icon Hover', 'socitycore' ),
			]
		);
		$this->add_control(
			'socity_cbox_ficon_hcolor',
			[
				'label' => esc_html__( 'Icon Hover Color', 'socitycore' ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000000',
				'selectors' => [
					'{{WRAPPER}} .static-contact-box:hover .cbox-icon i:before' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->add_control(
			'line2',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);
		$this->add_control(
			'socity_note_sms1',
			[
				'label' => __( '<strong>Image ICON</string>', 'socitycore' ),
				'type' => \Elementor\Controls_Manager::RAW_HTML,
				'content_classes' => 'note-message',
			]
		);
		$this->add_responsive_control(
			'socity_cbox_iimg_icon_size',
			[
				'label' => esc_html__( 'Image Icon Size', 'socitycore' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 65,
						'max' => 200,
						'step' => 1,
					],
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 17,
				],
				'selectors' => [
					'{{WRAPPER}} .cbox-icon img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
	
		$this->end_controls_section();
		$this->start_controls_section(
			'socity_cbox_title_style',
			[
			    'label' => esc_html__( 'Title Style', 'socitycore' ),
			    'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'socity_cbox_title_color',
			[
				'label' => esc_html__( 'Color', 'socitycore' ),
				'type' => Controls_Manager::COLOR,
				'default'	=>'#0a5be0',
				'selectors' => [
					'{{WRAPPER}} .cbox-title h2' => 'color: {{VALUE}};',
				],
			]
		); 
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'socity_cbox_title_typro',
				'selector' => '{{WRAPPER}} .cbox-title h2',
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'socity_cbox_dec_style',
			[
			    'label' => esc_html__( 'Content Style', 'socitycore' ),
			    'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'socity_cbox_con_color',
			[
				'label' => esc_html__( 'Color', 'socitycore' ),
				'type' => Controls_Manager::COLOR,
				'default'	=>'#798795',
				'selectors' => [
					'{{WRAPPER}} .cbox-dec' => 'color: {{VALUE}};',
				],
			]
		); 
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'socity_cbox_dec_typro',
				'selector' => '{{WRAPPER}} .cbox-dec p,.cbox-dec ul li',
			]
		);
		$this->add_responsive_control(
			'socity_cbox_title_margin',
			[
			    'label' => __( 'Margin', 'socitycore' ),
			    'type' => Controls_Manager::DIMENSIONS,
			    'size_units' => [ 'px', '%', 'em' ],
			    'default' => [
				  'top' => '0',
				  'right' => '0',
				  'bottom' => '0',
				  'left' => '0',
				  'isLinked' => true
			    ],
			    'selectors' => [
				  '{{WRAPPER}} .cbox-dec' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
			    ],
			    'separator' =>'before',
			]
		);
		$this->add_responsive_control(
			'socity_cbox_title_padding',
			[
			    'label' => __( 'padding', 'socitycore' ),
			    'type' => Controls_Manager::DIMENSIONS,
			    'size_units' => [ 'px', '%', 'em' ],
			    'default' => [
				  'top' => '0',
				  'right' => '38',
				  'bottom' => '0',
				  'left' => '38',
				  'isLinked' => true
			    ],
			    'selectors' => [
				  '{{WRAPPER}} .cbox-dec' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
			    ],
			    'separator' =>'before',
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render progress widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		?>
		<div class="static-contact-box">
			<div class="cbox-icon">
				<?php
					if(!empty($settings['socity_cbox_icon_type'] == 'img' )){
						echo Group_Control_Image_Size::get_attachment_image_html( $settings, 'socity_cbox_icon_img_resize', 'socity_cbox_icon_img' );
					}else{
						echo '<i class="'.esc_attr($settings['socity_cbox_icon']).'"></i>';
					}
				?>
			</div>
			<div class="cbox-title">
				<h2><?php echo esc_html($settings['socity_cbox_title']) ?></h2>
			</div>
			<div class="cbox-dec">
				<?php echo wp_kses_post(wpautop($settings['socity_cbox_content'])); ?>
			</div>
		</div>
		<?php
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new socity_contactbox_Widget );